<?php
require_once '../config/db.php';
require_once '../includes/auth_check.php';
require_once '../includes/db_connect.php';
require_once '../includes/functions.php';
require_once '../helpers/favicon_helper.php';
require_once '../helpers/permission_modal.php';

// Check if user has permission to view sales
// Check if user has any sales-related permissions
$salesPermissions = [
    'view_sales', 'process_sale', 'refund_sale', 'void_sale', 'view_customers'
];

if (!hasAnyPermission($salesPermissions)) {
    $_SESSION['error'] = "You don't have permission to access sales.";
    header('Location: dashboard.php');
    exit;
}

if (false) { // Disable the old check
    // Permission modal will be shown instead of redirecting
}

$title = "New Sale";
include '../includes/header.php';

// Get all products
$stmt = $pdo->query("SELECT id, name, selling_price FROM products WHERE quantity > 0 ORDER BY name");
$products = $stmt->fetchAll();

// Update the query to join with product_serial_numbers
$query = "SELECT s.*, p.name as product_name, p.price, psn.serial_number 
          FROM sales s 
          JOIN products p ON s.product_id = p.id 
          LEFT JOIN product_serial_numbers psn ON s.serial_number_id = psn.id 
          WHERE s.date BETWEEN ? AND ?";
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Sales - POS System</title>
    <?php echo getFaviconLink($pdo); ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
</head>

<div class="container mt-4">
    <h2>New Sale</h2>
    <form id="saleForm" method="post">
        <div class="row">
            <div class="col-md-6">
                <div class="form-group">
                    <label for="customer_name">Customer Name</label>
                    <input type="text" class="form-control" id="customer_name" name="customer_name" required>
                </div>
                <div class="form-group">
                    <label for="customer_phone">Customer Phone</label>
                    <input type="text" class="form-control" id="customer_phone" name="customer_phone">
                </div>
            </div>
            <div class="col-md-6">
                <div class="form-group">
                    <label for="payment_method">Payment Method</label>
                    <select class="form-control" id="payment_method" name="payment_method" required>
                        <option value="cash">Cash</option>
                        <option value="card">Card</option>
                        <option value="bank_transfer">Bank Transfer</option>
                    </select>
                </div>
                <div class="form-group">
                    <label for="notes">Notes</label>
                    <textarea class="form-control" id="notes" name="notes"></textarea>
                </div>
            </div>
        </div>

        <div class="card mt-4">
            <div class="card-header">
                <h4>Add Items</h4>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="product_id">Product</label>
                            <select class="form-control" id="product_id" required>
                                <option value="">Select Product</option>
                                <?php foreach ($products as $product): ?>
                                <option value="<?php echo $product['id']; ?>" data-price="<?php echo $product['selling_price']; ?>">
                                    <?php echo htmlspecialchars($product['name']); ?>
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>
                    <div class="col-md-2">
                        <div class="form-group">
                            <label for="quantity">Quantity</label>
                            <input type="number" class="form-control" id="quantity" min="1" required>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="serial_numbers">Serial Numbers</label>
                            <input type="text" class="form-control" id="serial_numbers" placeholder="Enter serial numbers separated by commas">
                            <small class="form-text text-muted">Enter serial numbers for each item, separated by commas</small>
                            <div id="available_serials" class="form-text text-info mt-2" style="display: none;"></div>
                        </div>
                    </div>
                    <div class="col-md-2">
                        <div class="form-group">
                            <label for="unit_price">Unit Price</label>
                            <input type="number" class="form-control" id="unit_price" step="0.01" required>
                        </div>
                    </div>
                    <div class="col-md-2">
                        <div class="form-group">
                            <label>&nbsp;</label>
                            <button type="button" class="btn btn-primary btn-block" onclick="addItem()">Add Item</button>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="card mt-4">
            <div class="card-header">
                <h4>Items</h4>
            </div>
            <div class="card-body">
                <table class="table" id="itemsTable">
                    <thead>
                        <tr>
                            <th>Product</th>
                            <th>Quantity</th>
                            <th>Serial Numbers</th>
                            <th>Unit Price</th>
                            <th>Total</th>
                            <th>Action</th>
                        </tr>
                    </thead>
                    <tbody></tbody>
                </table>
            </div>
        </div>

        <div class="row mt-4">
            <div class="col-md-6">
                <div class="form-group">
                    <label for="total_amount">Total Amount</label>
                    <input type="number" class="form-control" id="total_amount" name="total_amount" readonly>
                </div>
            </div>
            <div class="col-md-6 text-right">
                <button type="submit" class="btn btn-success">Complete Sale</button>
                <a href="sales.php" class="btn btn-secondary">Cancel</a>
            </div>
        </div>
    </form>
</div>

<script>
let items = [];

function addItem() {
    const productId = $('#product_id').val();
    const quantity = parseInt($('#quantity').val());
    const unitPrice = parseFloat($('#unit_price').val());
    const serialNumbers = $('#serial_numbers').val();
    
    if (!productId || !quantity || !unitPrice) {
        alert('Please fill in all required fields');
        return;
    }

    // Validate serial numbers if provided
    if (serialNumbers) {
        const serials = serialNumbers.split(',').map(s => s.trim()).filter(s => s);
        if (serials.length !== quantity) {
            alert('Number of serial numbers must match the quantity');
            return;
        }
    }

    const totalPrice = quantity * unitPrice;
    const productName = $('#product_id option:selected').text();
    
    const item = {
        product_id: productId,
        product_name: productName,
        quantity: quantity,
        unit_price: unitPrice,
        total_price: totalPrice,
        serial_numbers: serialNumbers
    };
    
    items.push(item);
    updateItemsTable();
    clearItemForm();
}

function removeItem(index) {
    items.splice(index, 1);
    updateItemsTable();
}

function updateItemsTable() {
    const tbody = $('#itemsTable tbody');
    tbody.empty();
    
    let total = 0;
    
    items.forEach((item, index) => {
        const row = `
            <tr>
                <td>${item.product_name}</td>
                <td>${item.quantity}</td>
                <td>${item.serial_numbers || '-'}</td>
                <td>${item.unit_price.toFixed(2)}</td>
                <td>${item.total_price.toFixed(2)}</td>
                <td>
                    <button type="button" class="btn btn-danger btn-sm" onclick="removeItem(${index})">
                        Remove
                    </button>
                </td>
            </tr>
        `;
        tbody.append(row);
        total += item.total_price;
    });
    
    $('#total_amount').val(total.toFixed(2));
}

function clearItemForm() {
    $('#product_id').val('');
    $('#quantity').val('');
    $('#unit_price').val('');
    $('#serial_numbers').val('');
    $('#available_serials').hide();
}

$(document).ready(function() {
    $('#product_id').change(function() {
        const price = $(this).find(':selected').data('price');
        $('#unit_price').val(price);
        
        // Fetch available serial numbers
        const productId = $(this).val();
        if (productId) {
            $.get('get_available_serials.php', { product_id: productId })
                .done(function(response) {
                    if (response.success && response.serials.length > 0) {
                        const serialsList = response.serials.join(', ');
                        $('#available_serials').html(`Available serial numbers: ${serialsList}`).show();
                    } else {
                        $('#available_serials').hide();
                    }
                })
                .fail(function() {
                    $('#available_serials').hide();
                });
        } else {
            $('#available_serials').hide();
        }
    });

    $('#saleForm').submit(function(e) {
        e.preventDefault();
        
        if (items.length === 0) {
            alert('Please add at least one item');
            return;
        }

        const formData = {
            customer_name: $('#customer_name').val(),
            customer_phone: $('#customer_phone').val(),
            payment_method: $('#payment_method').val(),
            notes: $('#notes').val(),
            total_amount: $('#total_amount').val(),
            items: items
        };

        $.ajax({
            url: 'process_sale.php',
            method: 'POST',
            data: JSON.stringify(formData),
            contentType: 'application/json',
            success: function(response) {
                if (response.success) {
                    alert('Sale completed successfully');
                    window.location.href = 'sales.php';
                } else {
                    alert('Error: ' + response.message);
                }
            },
            error: function() {
                alert('An error occurred while processing the sale');
            }
        });
    });
});
</script>

<?php 
// Add permission modal components
echo getPermissionModalCSS();
echo getPermissionModalHTML();
echo getPermissionModalJS();
?>

<?php echo getUniversalPermissionModalHTML(); ?>
<?php echo getUniversalPermissionModalJS(); ?>
</body>
</html> 